----------------------------------------- JCSlashCmd -----------------------------------------

--[[
	-- return the name of the class
	string JCSlashCmd:className()

	-- set an error handle for command errors (not really necessary)
	bool JCSlashCmd:setErrorHandler(fun)

	-- add an alias to the main slash command (ie /jubeitest has alias /jt)
	bool JCSlashCmd:addAlias(alias)

	-- add a command with required params, help info and function handler
	-- if info is nil, then the command won't be listed in showHelp
	-- nil info can also be used to create aliases to commands
	bool JCSlashCmd:addCmd(cmd,nparams,info,fun)

	-- get the slash command for this object
	string JCSlashCmd:getSlashCommand()

	-- get the error string associated with the error number
	string JCSlashCmd:getErrorString(error)

	-- if available, show the help messages
	bool JCSlashCmd:showHelp(msg)

	-- set the test to display before the individual command help
	bool JCSlashCmd:setHelpTitle(title)

	-- set the color to display the help message in
	bool JCSlashCmd:setHelpColor(red,green,blue)

	-- set a string to show if they call /slashcmd version
	bool JCSlashCmd:setHelpVersion(version)

	-- if available show the version string
	bool JCSlashCmd:showHelpVersion(version)

	-- INTERNAL: called by the SlashCmd system
	nil JCSlashCmd:proccessMsg(msg)
]]

JCSLASHCMD_ERROR_NONE = 1;
JCSLASHCMD_ERROR_UNKNOWN = 2;
JCSLASHCMD_ERROR_NUMPARAMS = 3;
JCSLASHCMD_ERROR_INVALID_CMD = 4;
JCSLASHCMD_ERROR_MISSING_CMD = 5;

JCSlashCmdErrorStrs = {
	"No error",
	"Unknown error",
	"Missing parameters",
	"Invalid command",
	"Missing command",
};

JCSlashCmd = {};
JCSlashCmd_mt = { __index = JCSlashCmd }

function JCSlashCmd:new(cmd_str)
	if (not cmd_str) then
		return nil;
	end

	local new_inst = {};
	setmetatable(new_inst, JCSlashCmd_mt);

	cmd_str = string.upper(cmd_str);

	new_inst.cmd_str = cmd_str;
	new_inst.class_name = "JCSlashCmd";
	new_inst.cmd_list = {};
	new_inst.errorHandler = nil;
	new_inst.error_show_help = true;
	new_inst.aliases = 1;
	new_inst.help_title = nil;
	new_inst.help_color = {1.0,1.0,1.0};
	new_inst.help_version = nil;

	setglobal("SLASH_"..string.upper(cmd_str)..new_inst.aliases,"/"..string.lower(cmd_str))

	SlashCmdList[cmd_str] = function(msg) new_inst:proccessMsg(msg) end;


	return new_inst;
end

function JCSlashCmd:className()
	return self.class_name;
end

function JCSlashCmd:getSlashCommand()
	return self.cmd_str;
end

function JCSlashCmd:setErrorHandler(fun)
	if (not fun) then return false end;

	self.errorHandler = fun;
	self.error_show_help = false;

	return true;
end

function JCSlashCmd:addAlias(alias)
	self.aliases = self.aliases + 1;
	setglobal("SLASH_"..string.upper(self.cmd_str)..self.aliases,"/"..string.lower(alias));

	return true;
end

function JCSlashCmd:addCmd(cmd,nparams,info,fun)
	if (self.cmd_list[cmd]) then return false end;
	if (not fun) then return false end;

	cmd = string.upper(cmd);

	self.cmd_list[cmd] = {};
	self.cmd_list[cmd].nparams = nparams;
	self.cmd_list[cmd].fun = fun;
	self.cmd_list[cmd].info = info;

	if (not self.cmd_list[cmd].nparams) then
		self.cmd_list[cmd].nparams = 0;
	end

--	table.setn(self.cmd_list,table.getn(self.cmd_list)+1)
--	ChatFrame1:AddMessage(cmd.." : "..self.cmd_list[cmd].info)

	return true;
end

function JCSlashCmd:getErrorString(error)
	if (not JCSlashCmdErrorStrs[error]) then return JCSlashCmdErrorStrs[JCSLASHCMD_ERROR_UNKNOWN] end;
	return JCSlashCmdErrorStrs[error];
end

function JCSlashCmd:showHelp(cmd)
	if not (cmd) then
		DEFAULT_CHAT_FRAME:AddMessage(self.help_title,unpack(self.help_color));

		-- TODO: make sure it list them in the order they were added

		foreach(self.cmd_list,function (k) if (self.cmd_list[k].info) then DEFAULT_CHAT_FRAME:AddMessage("/"..string.lower(self.cmd_str).." "..string.lower(k)..self.cmd_list[k].info, unpack(self.help_color)) end end);
--		foreach(self.cmd_list,function (k) if (self.cmd_list[k].info) then DEFAULT_CHAT_FRAME:AddMessage("/"..string.lower(self.cmd_str).." "..string.lower(k).." - "..self.cmd_list[k].info, unpack(self.help_color)) end end);

		if (self.help_version) then
			DEFAULT_CHAT_FRAME:AddMessage("/"..string.lower(self.cmd_str).." ".."version - show UI mod version", unpack(self.help_color))
		end
	else
		cmd = string.upper(cmd);
		if (self.cmd_list[cmd] and self.cmd_list[cmd].info) then
			DEFAULT_CHAT_FRAME:AddMessage("/"..string.lower(self.cmd_str).." "..string.lower(cmd).." - "..self.cmd_list[cmd].info, unpack(self.help_color));
		end
	end

	return true;
end

function JCSlashCmd:setHelpTitle(title)
	self.help_title = title;

	return true;
end

function JCSlashCmd:setHelpColor(red,green,blue)
	self.help_color = {red,green,blue};
	return true;
end

function JCSlashCmd:setHelpVersion(version)
	self.help_version = version;
	return true;
end

function JCSlashCmd:showHelpVersion(version)
	DEFAULT_CHAT_FRAME:AddMessage(self.help_version,unpack(self.help_color));
	return true;
end

function JCSlashCmd:proccessMsg(msg_str)
	local args = {};
	local cmd = nil;
	local word;
	local i = 1;

	local args_str = string.match(msg_str,"%S+%s(.*)")

	msg = string.gsub(msg_str,","," ");

	for word in string.gmatch(msg, "%S+") do
		args[i] = word;
		i = i + 1;
	end

	if (i == 1) then
		if (self.errorHandler) then
			self.errorHandler(self,JCSLASHCMD_ERROR_MISSING_CMD,self.cmd_str,msg_str);
		elseif (self.error_show_help) then
			self:showHelp();
		end
		return;
	end

	cmd = string.upper(args[1]);

	if (not self.cmd_list[cmd]) then
		if ((self.help_version) and (cmd == "VERSION")) then
			self:showHelpVersion();
			return;
		end

		if (self.errorHandler) then
			self.errorHandler(self,JCSLASHCMD_ERROR_INVALID_CMD,self.cmd_str,msg_str);
		elseif (self.error_show_help) then
			self:showHelp();
		end
		return;
	end

	table.remove(args,1);

	if ((i-2) < self.cmd_list[cmd].nparams) then
		if (self.errorHandler) then
			self.errorHandler(self,JCSLASHCMD_ERROR_NUMPARAMS);
		elseif (self.error_show_help) then
			self:showHelp(cmd);
		end
		return;
	end

	self.cmd_list[cmd].fun(args,args_str)
end

